/**
  ******************************************************************************
  * @file    usart_common_interface.c
  * @author  MCD Application Team
  * @brief   Contains USART HW configuration
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2023 Puya Semiconductor Co.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by Puya under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  */

#include "usart_common_interface.h"
#include "iwdg_interface.h"
#include "wwdg_interface.h"
#include "openbl_usart_cmd.h"

#define DEFAULT_BAUDRATE                  115200            /* Default Baudrate */
#define DEFAULT_BRR                       ((SystemCoreClock+DEFAULT_BAUDRATE/2)/DEFAULT_BAUDRATE)
void USART_Init(USART_TypeDef * usart)
{
//  SET_BIT(RCC->APBENR2, RCC_APBENR2_USART1EN);
  
  //USART_CR1_M  1： 1 start bit， 9 data bit， n stop bit
  //USART_CR1_PCE  1：奇偶校验使能
  //USART_CR1_PS  0：偶校验(EVEN)
  //USART_CR2_STOP 00： 1 stop bit
  SET_BIT(usart->CR1, (USART_CR1_M | USART_CR1_PCE));

  WRITE_REG(usart->BRR, DEFAULT_BRR);

  SET_BIT(usart->CR1, USART_CR1_UE);//1： USART 使能

  //USART_CR3_ABRMODE 00：从 start 位开始测量波特率
  //USART_CR3_ABREN 1：自动波特率使能
//  MODIFY_REG(usart->CR3, (USART_CR3_ABRMOD | USART_CR3_ABREN), USART_CR3_ABREN);

  SET_BIT(usart->CR1, USART_CR1_RE | USART_CR1_TE); //1： 收发使能
  
}

void USART_DeInit(USART_TypeDef * usart)
{
  usart->CR1 &= ~USART_CR1_UE;
//  HAL_SCI_DeInit(husart);
}

void USART_Read(USART_TypeDef * usart,uint8_t *buf, uint16_t size)
{
  while (size--)
  {
    while (USART_SR_RXNE != READ_BIT(usart->SR, USART_SR_RXNE))
    {
      OPENBL_WWDG_Refresh();
      OPENBL_IWDG_Refresh();
    }

    *buf++ = (uint8_t)(READ_BIT(usart->DR, USART_DR_DR) & 0xFFU);
  }
}

void USART_Write(USART_TypeDef * usart,uint8_t *buf, uint16_t size)
{
  while (size--)
  {
    while (USART_SR_TXE != READ_BIT(usart->SR, USART_SR_TXE))
    {
      OPENBL_WWDG_Refresh();
      OPENBL_IWDG_Refresh();
    }

    usart->DR = ((*buf++) & 0x0FF);
  }

  while (USART_SR_TC != READ_BIT(usart->SR, USART_SR_TC))
  {
    OPENBL_WWDG_Refresh();
    OPENBL_IWDG_Refresh();
  }

  CLEAR_BIT(usart->SR, USART_SR_TC);
}

uint8_t USART_ProtocolDetection(USART_TypeDef * usart)
{
  uint8_t tmp = 0U;

  OPENBL_WWDG_Refresh();
  OPENBL_IWDG_Refresh();

  tmp = usart->DR & 0xFFU;

  /* Check if the USART is addressed */
  if (SYNC_BYTE == tmp)
  {
    CLEAR_BIT(usart->CR3, USART_CR3_ABREN);

    /* Aknowledge the host */
    tmp = ACK_BYTE;
    USART_Write(usart, &tmp, 1);

    return 1;
  }

  (usart->SR |= (uint16_t)(USART_SR_ABRRQ));

  return 0;
}

/************************ (C) COPYRIGHT Puya *****END OF FILE****/
